#!/usr/bin/env python3
import os, yaml, numpy as np, pandas as pd
import matplotlib.pyplot as plt

# Configuration
workspace    = os.getcwd()
flip_dir     = os.path.join(workspace, '..', 'vol4-flip-count-simulator_updated')
wilson_dir   = workspace
results_dir  = os.path.join(wilson_dir, 'results_crossover')
os.makedirs(results_dir, exist_ok=True)

# 1) Extend loop sizes to get more data points
loop_sizes = list(range(1, 11))   # loops of edge 1…10
# Update config.yaml
cfg_path = os.path.join(wilson_dir, 'config.yaml')
cfg = yaml.safe_load(open(cfg_path))
cfg['loop_sizes'] = loop_sizes
with open(cfg_path, 'w') as f:
    yaml.safe_dump(cfg, f)

# 2) Regenerate flip counts and kernel
os.system(f"conda activate flip-x && python {os.path.join(flip_dir,'scripts/generate_flip_counts.py')}")
os.system(f"conda activate wilson-x && python scripts/make_kernel_from_eigs.py")

# 3) Run full simulation with new loop sizes
os.system(f"conda activate wilson-x && python src/run_simulation.py --config config.yaml")

# 4) For each gauge group compute fits and crossover
summary = []
for G in ['U1','SU2','SU3']:
    csvf = os.path.join(wilson_dir, 'results', f'wilson_{G}.csv')
    df   = pd.read_csv(csvf)
    avg  = df['real'] + 1j*df['imag']
    log_avg = np.log(np.abs(avg))
    areas    = np.array(loop_sizes)**2
    perims   = 4 * np.array(loop_sizes)

    # fit linear models
    m_area, c_area = np.polyfit(areas, log_avg, 1)
    m_peri, c_peri = np.polyfit(perims, log_avg, 1)

    # solve m_area*L^2 + c_area = m_peri*(4L) + c_peri
    # => m_area*L^2 - 4*m_peri*L + (c_area - c_peri) = 0
    a, b, d = m_area, -4*m_peri, (c_area - c_peri)
    disc = b**2 - 4*a*d
    L_cross = None
    if disc >= 0 and a != 0:
        roots = [(-b + np.sqrt(disc))/(2*a), (-b - np.sqrt(disc))/(2*a)]
        # choose positive real root within loop_sizes range
        for r in roots:
            if r > 0 and r <= max(loop_sizes):
                L_cross = float(r)
                break

    # save to summary
    summary.append({
        'Gauge': G,
        'Slope_Area': float(m_area),
        'Slope_Perimeter': float(m_peri),
        'Crossover_LoopSize': L_cross
    })

    # 5) Plot both data and fits on same figure
    plt.figure()
    plt.scatter(areas, log_avg, label='data')
    xs = np.linspace(min(areas), max(areas), 200)
    plt.plot(xs, m_area*xs + c_area, '--', label='Area fit')
    pp = np.linspace(min(perims), max(perims), 200)
    plt.plot(pp, m_peri*pp + c_peri, '-.', label='Perimeter fit')
    plt.xlabel('X')
    plt.ylabel('ln(|<W>|)')
    plt.title(f'Crossover Analysis for {G}')
    plt.legend()
    plt.savefig(os.path.join(results_dir, f'crossover_{G}.png'))
    plt.close()

# 6) Write summary CSV and report
import csv
out_csv = os.path.join(results_dir, 'crossover_summary.csv')
with open(out_csv, 'w', newline='') as f:
    writer = csv.DictWriter(f, fieldnames=summary[0].keys())
    writer.writeheader()
    writer.writerows(summary)

# Markdown report
report = ["# Area vs Perimeter Crossover Report\n"]
for row in summary:
    report.append(f"## {row['Gauge']}\n"
                  f"- Area‐law slope: {row['Slope_Area']:.5f}\n"
                  f"- Perimeter‐law slope: {row['Slope_Perimeter']:.5f}\n"
                  f"- Estimated crossover loop size: {row['Crossover_LoopSize']}\n"
                  f"![Crossover Plot]({os.path.basename(results_dir)}/crossover_{row['Gauge']}.png)\n")
with open(os.path.join(results_dir, 'crossover_report.md'), 'w') as f:
    f.write('\n'.join(report))